/**
 * FlipBook Magazine - Enhanced Edition
 * Main Application JavaScript
 */

class FlipBookApp {
    constructor() {
        this.currentView = 'library';
        this.currentMagazine = null;
        this.magazines = [];
        this.flipbook = null;
        this.currentPage = 1;
        this.totalPages = 0;
        this.zoomLevel = 1;
        
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadMagazines();
        this.setupDragAndDrop();
    }

    bindEvents() {
        // Navigation buttons
        document.getElementById('upload-btn').addEventListener('click', () => this.showUploadModal());
        document.getElementById('library-btn').addEventListener('click', () => this.showLibrary());
        document.getElementById('back-to-library').addEventListener('click', () => this.showLibrary());

        // Modal events
        document.getElementById('close-modal').addEventListener('click', () => this.hideUploadModal());
        document.getElementById('upload-modal').addEventListener('click', (e) => {
            if (e.target.id === 'upload-modal') this.hideUploadModal();
        });

        // File input
        document.getElementById('pdf-input').addEventListener('change', (e) => this.handleFileSelect(e));

        // Reader controls
        document.getElementById('prev-page').addEventListener('click', () => this.previousPage());
        document.getElementById('next-page').addEventListener('click', () => this.nextPage());
        document.getElementById('first-page').addEventListener('click', () => this.goToPage(1));
        document.getElementById('last-page').addEventListener('click', () => this.goToPage(this.totalPages));
        
        document.getElementById('zoom-in').addEventListener('click', () => this.zoomIn());
        document.getElementById('zoom-out').addEventListener('click', () => this.zoomOut());
        document.getElementById('fullscreen-btn').addEventListener('click', () => this.toggleFullscreen());
        document.getElementById('download-pdf').addEventListener('click', () => this.downloadCurrentPDF());

        // Search and sort
        document.getElementById('search-input').addEventListener('input', (e) => this.searchMagazines(e.target.value));
        document.getElementById('sort-select').addEventListener('change', (e) => this.sortMagazines(e.target.value));

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => this.handleKeyboard(e));
    }

    setupDragAndDrop() {
        const uploadArea = document.getElementById('upload-area');
        
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, this.preventDefaults, false);
        });

        ['dragenter', 'dragover'].forEach(eventName => {
            uploadArea.addEventListener(eventName, () => uploadArea.classList.add('dragover'), false);
        });

        ['dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, () => uploadArea.classList.remove('dragover'), false);
        });

        uploadArea.addEventListener('drop', (e) => this.handleDrop(e), false);
        uploadArea.addEventListener('click', () => document.getElementById('pdf-input').click());
    }

    preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }

    handleDrop(e) {
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            this.handleFile(files[0]);
        }
    }

    handleFileSelect(e) {
        const file = e.target.files[0];
        if (file) {
            this.handleFile(file);
        }
    }

    handleFile(file) {
        if (file.type !== 'application/pdf') {
            this.showToast('Please select a PDF file', 'error');
            return;
        }

        if (file.size > 50 * 1024 * 1024) { // 50MB limit
            this.showToast('File size must be less than 50MB', 'error');
            return;
        }

        this.uploadFile(file);
    }

    async uploadFile(file) {
        this.showUploadProgress();
        
        try {
            // Step 1: Upload
            this.updateProgress(25, 'Uploading file...');
            await this.simulateDelay(1000);
            
            const formData = new FormData();
            formData.append('pdf', file);
            
            const response = await fetch('api/upload.php', {
                method: 'POST',
                body: formData
            });
            
            if (!response.ok) {
                throw new Error('Upload failed');
            }
            
            // Step 2: Process
            this.updateProgress(50, 'Processing PDF...');
            await this.simulateDelay(1500);
            
            // Step 3: Convert
            this.updateProgress(75, 'Converting to flipbook...');
            await this.simulateDelay(2000);
            
            // Step 4: Complete
            this.updateProgress(100, 'Complete!');
            await this.simulateDelay(500);
            
            this.showUploadResult(true, 'Magazine uploaded successfully!');
            this.loadMagazines(); // Refresh the library
            
        } catch (error) {
            console.error('Upload error:', error);
            this.showUploadResult(false, 'Upload failed. Please try again.');
        }
    }

    showUploadProgress() {
        document.getElementById('upload-area').style.display = 'none';
        document.getElementById('upload-progress').style.display = 'block';
        document.getElementById('upload-result').style.display = 'none';
    }

    updateProgress(percent, status) {
        document.getElementById('progress-fill').style.width = percent + '%';
        document.getElementById('progress-status').textContent = status;
        
        // Update steps
        const steps = document.querySelectorAll('.step');
        steps.forEach((step, index) => {
            if (percent >= (index + 1) * 25) {
                step.classList.add('complete');
                step.classList.remove('active');
            } else if (percent >= index * 25) {
                step.classList.add('active');
            }
        });
    }

    showUploadResult(success, message) {
        document.getElementById('upload-progress').style.display = 'none';
        document.getElementById('upload-result').style.display = 'block';
        document.getElementById('upload-result').innerHTML = `
            <div class="result-icon">${success ? '✅' : '❌'}</div>
            <h4>${success ? 'Success!' : 'Error'}</h4>
            <p>${message}</p>
            <button class="btn btn-primary" onclick="flipbookApp.hideUploadModal()">
                ${success ? 'View Library' : 'Try Again'}
            </button>
        `;
    }

    simulateDelay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    async loadMagazines() {
        try {
            const response = await fetch('?api=magazines');
            const data = await response.json();
            this.magazines = data.magazines || [];
            this.renderMagazines();
        } catch (error) {
            console.error('Failed to load magazines:', error);
            this.magazines = [];
            this.renderMagazines();
        }
    }

    renderMagazines() {
        const grid = document.getElementById('magazine-grid');
        const emptyLibrary = document.getElementById('empty-library');
        
        if (this.magazines.length === 0) {
            grid.style.display = 'none';
            emptyLibrary.style.display = 'block';
            return;
        }
        
        grid.style.display = 'grid';
        emptyLibrary.style.display = 'none';
        
        grid.innerHTML = this.magazines.map(magazine => `
            <div class="magazine-card" onclick="flipbookApp.openMagazine('${magazine.id}')">
                <div class="magazine-cover">
                    📖
                </div>
                <div class="magazine-info">
                    <div class="magazine-title">${this.escapeHtml(magazine.title)}</div>
                    <div class="magazine-meta">
                        ${this.formatFileSize(magazine.size)} • ${this.formatDate(magazine.date)}
                    </div>
                    <div class="magazine-actions" onclick="event.stopPropagation()">
                        <button class="btn btn-primary" onclick="flipbookApp.openMagazine('${magazine.id}')">
                            📖 Read
                        </button>
                        <button class="btn btn-secondary" onclick="flipbookApp.downloadMagazine('${magazine.id}')">
                            📥 Download
                        </button>
                    </div>
                </div>
            </div>
        `).join('');
    }

    searchMagazines(query) {
        const filteredMagazines = this.magazines.filter(magazine =>
            magazine.title.toLowerCase().includes(query.toLowerCase())
        );
        this.renderFilteredMagazines(filteredMagazines);
    }

    sortMagazines(sortBy) {
        let sorted = [...this.magazines];
        
        switch (sortBy) {
            case 'newest':
                sorted.sort((a, b) => b.date - a.date);
                break;
            case 'oldest':
                sorted.sort((a, b) => a.date - b.date);
                break;
            case 'title':
                sorted.sort((a, b) => a.title.localeCompare(b.title));
                break;
            case 'size':
                sorted.sort((a, b) => b.size - a.size);
                break;
        }
        
        this.magazines = sorted;
        this.renderMagazines();
    }

    renderFilteredMagazines(magazines) {
        const grid = document.getElementById('magazine-grid');
        
        if (magazines.length === 0) {
            grid.innerHTML = '<div class="no-results">No magazines found</div>';
            return;
        }
        
        grid.innerHTML = magazines.map(magazine => `
            <div class="magazine-card" onclick="flipbookApp.openMagazine('${magazine.id}')">
                <div class="magazine-cover">📖</div>
                <div class="magazine-info">
                    <div class="magazine-title">${this.escapeHtml(magazine.title)}</div>
                    <div class="magazine-meta">
                        ${this.formatFileSize(magazine.size)} • ${this.formatDate(magazine.date)}
                    </div>
                    <div class="magazine-actions" onclick="event.stopPropagation()">
                        <button class="btn btn-primary" onclick="flipbookApp.openMagazine('${magazine.id}')">
                            📖 Read
                        </button>
                        <button class="btn btn-secondary" onclick="flipbookApp.downloadMagazine('${magazine.id}')">
                            📥 Download
                        </button>
                    </div>
                </div>
            </div>
        `).join('');
    }

    async openMagazine(magazineId) {
        const magazine = this.magazines.find(m => m.id === magazineId);
        if (!magazine) return;
        
        this.currentMagazine = magazine;
        this.showReader();
        
        // Update reader header
        document.getElementById('current-magazine-title').textContent = magazine.title;
        document.getElementById('current-magazine-info').textContent = 
            `${this.formatFileSize(magazine.size)} • ${this.formatDate(magazine.date)}`;
        
        // Load PDF and create flipbook
        await this.loadPDFAsFlipbook(magazine.filepath);
    }

    async loadPDFAsFlipbook(filepath) {
        try {
            const pdfUrl = filepath;
            
            // Load PDF using PDF.js
            const pdf = await pdfjsLib.getDocument(pdfUrl).promise;
            this.totalPages = pdf.numPages;
            
            // Create flipbook container
            const flipbookContainer = document.getElementById('flipbook');
            flipbookContainer.innerHTML = '';
            
            // Create pages
            for (let i = 1; i <= this.totalPages; i++) {
                const page = await pdf.getPage(i);
                const canvas = document.createElement('canvas');
                const context = canvas.getContext('2d');
                
                const viewport = page.getViewport({ scale: 1.5 });
                canvas.width = viewport.width;
                canvas.height = viewport.height;
                
                await page.render({ canvasContext: context, viewport: viewport }).promise;
                
                const pageDiv = document.createElement('div');
                pageDiv.className = 'turn-page';
                pageDiv.appendChild(canvas);
                flipbookContainer.appendChild(pageDiv);
            }
            
            // Initialize Turn.js
            $(flipbookContainer).turn({
                width: 800,
                height: 600,
                autoCenter: true,
                duration: 1000,
                gradients: true,
                when: {
                    turning: (event, page, view) => {
                        this.currentPage = page;
                        this.updatePageInfo();
                    },
                    turned: (event, page, view) => {
                        this.currentPage = page;
                        this.updatePageInfo();
                        
                        // Play page turn sound if enabled
                        if (document.getElementById('page-sounds').checked) {
                            this.playPageTurnSound();
                        }
                    }
                }
            });
            
            this.flipbook = $(flipbookContainer);
            this.currentPage = 1;
            this.updatePageInfo();
            
        } catch (error) {
            console.error('Failed to load PDF:', error);
            this.showToast('Failed to load magazine', 'error');
        }
    }

    updatePageInfo() {
        document.getElementById('page-info').textContent = `Page ${this.currentPage} of ${this.totalPages}`;
    }

    previousPage() {
        if (this.flipbook && this.currentPage > 1) {
            this.flipbook.turn('previous');
        }
    }

    nextPage() {
        if (this.flipbook && this.currentPage < this.totalPages) {
            this.flipbook.turn('next');
        }
    }

    goToPage(page) {
        if (this.flipbook && page >= 1 && page <= this.totalPages) {
            this.flipbook.turn('page', page);
        }
    }

    zoomIn() {
        this.zoomLevel = Math.min(this.zoomLevel * 1.2, 3);
        this.applyZoom();
    }

    zoomOut() {
        this.zoomLevel = Math.max(this.zoomLevel / 1.2, 0.5);
        this.applyZoom();
    }

    applyZoom() {
        if (this.flipbook) {
            const flipbookElement = this.flipbook[0];
            flipbookElement.style.transform = `scale(${this.zoomLevel})`;
        }
    }

    toggleFullscreen() {
        if (!document.fullscreenElement) {
            document.getElementById('reader-view').requestFullscreen();
        } else {
            document.exitFullscreen();
        }
    }

    downloadCurrentPDF() {
        if (this.currentMagazine) {
            this.downloadMagazine(this.currentMagazine.id);
        }
    }

    downloadMagazine(magazineId) {
        const magazine = this.magazines.find(m => m.id === magazineId);
        if (magazine) {
            const link = document.createElement('a');
            link.href = `uploads/${magazine.filename}`;
            link.download = magazine.filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    }

    playPageTurnSound() {
        // Create a subtle page turn sound effect
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);
        
        oscillator.frequency.setValueAtTime(200, audioContext.currentTime);
        oscillator.frequency.exponentialRampToValueAtTime(100, audioContext.currentTime + 0.1);
        
        gainNode.gain.setValueAtTime(0.1, audioContext.currentTime);
        gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.1);
        
        oscillator.start(audioContext.currentTime);
        oscillator.stop(audioContext.currentTime + 0.1);
    }

    handleKeyboard(e) {
        if (this.currentView === 'reader') {
            switch (e.key) {
                case 'ArrowLeft':
                    e.preventDefault();
                    this.previousPage();
                    break;
                case 'ArrowRight':
                    e.preventDefault();
                    this.nextPage();
                    break;
                case 'Home':
                    e.preventDefault();
                    this.goToPage(1);
                    break;
                case 'End':
                    e.preventDefault();
                    this.goToPage(this.totalPages);
                    break;
                case 'Escape':
                    this.showLibrary();
                    break;
                case '+':
                case '=':
                    e.preventDefault();
                    this.zoomIn();
                    break;
                case '-':
                    e.preventDefault();
                    this.zoomOut();
                    break;
            }
        }
    }

    showLibrary() {
        this.currentView = 'library';
        document.getElementById('library-view').style.display = 'block';
        document.getElementById('reader-view').style.display = 'none';
        
        if (this.flipbook) {
            this.flipbook.turn('destroy');
            this.flipbook = null;
        }
    }

    showReader() {
        this.currentView = 'reader';
        document.getElementById('library-view').style.display = 'none';
        document.getElementById('reader-view').style.display = 'block';
    }

    showUploadModal() {
        document.getElementById('upload-modal').classList.add('active');
        // Reset modal state
        document.getElementById('upload-area').style.display = 'block';
        document.getElementById('upload-progress').style.display = 'none';
        document.getElementById('upload-result').style.display = 'none';
        document.getElementById('pdf-input').value = '';
    }

    hideUploadModal() {
        document.getElementById('upload-modal').classList.remove('active');
    }

    showToast(message, type = 'success') {
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        toast.innerHTML = `
            <div class="toast-content">
                <strong>${type === 'error' ? 'Error' : type === 'warning' ? 'Warning' : 'Success'}</strong>
                <p>${message}</p>
            </div>
        `;
        
        document.getElementById('toast-container').appendChild(toast);
        
        setTimeout(() => {
            toast.remove();
        }, 5000);
    }

    // Utility functions
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    formatDate(timestamp) {
        return new Date(timestamp * 1000).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    }
}

// Global functions for onclick handlers
function showUploadModal() {
    window.flipbookApp.showUploadModal();
}
