<?php
// Simple Magazine Detection - This will definitely work!
function getExistingMagazines() {
    $magazines = [];
    
    // Check multiple possible upload directories
    $possibleDirs = [
        'uploads/',
        '../flipbook-simple/uploads/',
        '../flipbook-enhanced/uploads/',
        '../uploads/'
    ];
    
    foreach ($possibleDirs as $dir) {
        if (is_dir($dir)) {
            $files = glob($dir . "*.pdf");
            foreach ($files as $file) {
                $filename = basename($file);
                $magazines[] = [
                    'id' => md5($file),
                    'filename' => $filename,
                    'title' => str_replace(['.pdf', '_', '-'], [' ', ' ', ' '], pathinfo($filename, PATHINFO_FILENAME)),
                    'filepath' => $file,
                    'size' => filesize($file),
                    'date' => filemtime($file),
                    'url' => $file
                ];
            }
        }
    }
    
    // Remove duplicates by filename
    $unique = [];
    $seen = [];
    foreach ($magazines as $mag) {
        if (!in_array($mag['filename'], $seen)) {
            $unique[] = $mag;
            $seen[] = $mag['filename'];
        }
    }
    
    // Sort by date (newest first)
    usort($unique, function($a, $b) {
        return $b['date'] - $a['date'];
    });
    
    return $unique;
}

// Handle API requests
if (isset($_GET['api'])) {
    header('Content-Type: application/json');
    if ($_GET['api'] === 'magazines') {
        echo json_encode(['magazines' => getExistingMagazines()]);
        exit;
    }
}

$magazines = getExistingMagazines();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>FlipBook Magazine - Working Edition</title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.7.1/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/turn.js/3/turn.min.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: white;
        }
        
        .header {
            background: rgba(255,255,255,0.1);
            padding: 1rem 2rem;
            backdrop-filter: blur(10px);
        }
        
        .header h1 {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        .magazine-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 2rem;
            margin-top: 2rem;
        }
        
        .magazine-card {
            background: rgba(255,255,255,0.1);
            border-radius: 15px;
            padding: 1.5rem;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255,255,255,0.2);
            transition: transform 0.3s ease;
            cursor: pointer;
        }
        
        .magazine-card:hover {
            transform: translateY(-5px);
            background: rgba(255,255,255,0.15);
        }
        
        .magazine-cover {
            font-size: 4rem;
            text-align: center;
            margin-bottom: 1rem;
        }
        
        .magazine-title {
            font-size: 1.2rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
            text-transform: capitalize;
        }
        
        .magazine-meta {
            font-size: 0.9rem;
            opacity: 0.8;
            margin-bottom: 1rem;
        }
        
        .magazine-actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: bold;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: #4CAF50;
            color: white;
        }
        
        .btn-primary:hover {
            background: #45a049;
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: rgba(255,255,255,0.2);
            color: white;
        }
        
        .btn-secondary:hover {
            background: rgba(255,255,255,0.3);
        }
        
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
        }
        
        .empty-state h2 {
            font-size: 2rem;
            margin-bottom: 1rem;
        }
        
        .flipbook-reader {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.9);
            z-index: 1000;
        }
        
        .reader-header {
            background: rgba(255,255,255,0.1);
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .reader-content {
            display: flex;
            justify-content: center;
            align-items: center;
            height: calc(100vh - 120px);
            padding: 2rem;
        }
        
        .flipbook {
            margin: 0 auto;
        }
        
        .flipbook .turn-page {
            background: white;
            border: 1px solid #ddd;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }
        
        .flipbook .turn-page canvas {
            width: 100%;
            height: 100%;
            display: block;
        }
        
        .reader-controls {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            display: flex;
            gap: 1rem;
            background: rgba(0,0,0,0.7);
            padding: 1rem;
            border-radius: 10px;
        }
        
        @media (max-width: 768px) {
            .magazine-grid {
                grid-template-columns: 1fr;
            }
            
            .flipbook {
                width: 90vw !important;
                height: 60vh !important;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>📖 FlipBook Magazine</h1>
        <p>Working Edition - Real Flipbook Experience</p>
    </div>

    <div class="container">
        <?php if (empty($magazines)): ?>
            <div class="empty-state">
                <div style="font-size: 4rem; margin-bottom: 1rem;">📚</div>
                <h2>No magazines found</h2>
                <p>Upload some PDF magazines to get started!</p>
            </div>
        <?php else: ?>
            <h2>Your Magazine Library (<?php echo count($magazines); ?> magazines)</h2>
            <div class="magazine-grid">
                <?php foreach ($magazines as $magazine): ?>
                    <div class="magazine-card" onclick="openMagazine('<?php echo htmlspecialchars($magazine['id']); ?>')">
                        <div class="magazine-cover">📖</div>
                        <div class="magazine-title"><?php echo htmlspecialchars($magazine['title']); ?></div>
                        <div class="magazine-meta">
                            <?php echo number_format($magazine['size'] / 1024 / 1024, 1); ?> MB • 
                            <?php echo date('M j, Y', $magazine['date']); ?>
                        </div>
                        <div class="magazine-actions" onclick="event.stopPropagation()">
                            <button class="btn btn-primary" onclick="openMagazine('<?php echo htmlspecialchars($magazine['id']); ?>')">
                                📖 Read Flipbook
                            </button>
                            <a href="<?php echo htmlspecialchars($magazine['filepath']); ?>" class="btn btn-secondary" download>
                                📥 Download
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Flipbook Reader -->
    <div id="flipbook-reader" class="flipbook-reader">
        <div class="reader-header">
            <div>
                <h3 id="current-title">Magazine Title</h3>
                <p id="current-info">Loading...</p>
            </div>
            <button class="btn btn-secondary" onclick="closeReader()">✕ Close</button>
        </div>
        <div class="reader-content">
            <div id="flipbook" class="flipbook">
                <div style="text-align: center; padding: 2rem; color: white;">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">📖</div>
                    <h3>Loading magazine...</h3>
                    <p>Please wait while we prepare your flipbook experience</p>
                </div>
            </div>
        </div>
        <div class="reader-controls">
            <button class="btn btn-secondary" onclick="previousPage()">◀ Previous</button>
            <span id="page-info" style="color: white; padding: 0.5rem 1rem;">Page 1 of 1</span>
            <button class="btn btn-primary" onclick="nextPage()">Next ▶</button>
        </div>
    </div>

    <script>
        // Magazine data from PHP
        const magazines = <?php echo json_encode($magazines); ?>;
        let currentMagazine = null;
        let flipbook = null;
        let currentPage = 1;
        let totalPages = 0;

        // Set PDF.js worker
        pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

        function openMagazine(magazineId) {
            currentMagazine = magazines.find(m => m.id === magazineId);
            if (!currentMagazine) return;

            document.getElementById('current-title').textContent = currentMagazine.title;
            document.getElementById('current-info').textContent = 
                `${(currentMagazine.size / 1024 / 1024).toFixed(1)} MB • ${new Date(currentMagazine.date * 1000).toLocaleDateString()}`;

            document.getElementById('flipbook-reader').style.display = 'block';
            document.body.style.overflow = 'hidden';

            loadPDFAsFlipbook(currentMagazine.filepath);
        }

        function closeReader() {
            document.getElementById('flipbook-reader').style.display = 'none';
            document.body.style.overflow = 'auto';
            
            if (flipbook) {
                flipbook.turn('destroy');
                flipbook = null;
            }
        }

        async function loadPDFAsFlipbook(filepath) {
            try {
                const flipbookContainer = document.getElementById('flipbook');
                flipbookContainer.innerHTML = '<div style="text-align: center; padding: 2rem; color: white;"><h3>Loading pages...</h3></div>';

                const pdf = await pdfjsLib.getDocument(filepath).promise;
                totalPages = pdf.numPages;

                flipbookContainer.innerHTML = '';

                // Create pages
                for (let i = 1; i <= totalPages; i++) {
                    const page = await pdf.getPage(i);
                    const canvas = document.createElement('canvas');
                    const context = canvas.getContext('2d');

                    const viewport = page.getViewport({ scale: 1.5 });
                    canvas.width = viewport.width;
                    canvas.height = viewport.height;

                    await page.render({ canvasContext: context, viewport: viewport }).promise;

                    const pageDiv = document.createElement('div');
                    pageDiv.className = 'turn-page';
                    pageDiv.appendChild(canvas);
                    flipbookContainer.appendChild(pageDiv);
                }

                // Initialize Turn.js flipbook
                $(flipbookContainer).turn({
                    width: 800,
                    height: 600,
                    autoCenter: true,
                    duration: 1000,
                    gradients: true,
                    when: {
                        turning: function(event, page, view) {
                            currentPage = page;
                            updatePageInfo();
                        },
                        turned: function(event, page, view) {
                            currentPage = page;
                            updatePageInfo();
                        }
                    }
                });

                flipbook = $(flipbookContainer);
                currentPage = 1;
                updatePageInfo();

            } catch (error) {
                console.error('Failed to load PDF:', error);
                document.getElementById('flipbook').innerHTML = 
                    '<div style="text-align: center; padding: 2rem; color: white;"><h3>Error loading magazine</h3><p>Please try again or check if the file exists.</p></div>';
            }
        }

        function updatePageInfo() {
            document.getElementById('page-info').textContent = `Page ${currentPage} of ${totalPages}`;
        }

        function previousPage() {
            if (flipbook && currentPage > 1) {
                flipbook.turn('previous');
            }
        }

        function nextPage() {
            if (flipbook && currentPage < totalPages) {
                flipbook.turn('next');
            }
        }

        // Keyboard navigation
        document.addEventListener('keydown', function(e) {
            if (document.getElementById('flipbook-reader').style.display === 'block') {
                switch(e.key) {
                    case 'ArrowLeft':
                        e.preventDefault();
                        previousPage();
                        break;
                    case 'ArrowRight':
                        e.preventDefault();
                        nextPage();
                        break;
                    case 'Escape':
                        closeReader();
                        break;
                }
            }
        });
    </script>
</body>
</html>
